#!/bin/sh

APPSUPPORT_OVERLAY_VERSION=1
APPSUPPORT_OVERLAY_CACHE=/var/lib/appsupport_overlay/cache
APPSUPPORT_OVERLAY_CONFIG=/var/lib/appsupport_overlay/overlay_config
APPSUPPORT_OVERLAY_TAR="appsupport_overlay.tar.gz"
APPSUPPORT_OVERLAY_IMG_PRIVILEGED="appsupport_overlay_priv.img"
APPSUPPORT_OVERLAY_IMG_UNPRIVILEGED="appsupport_overlay.img"
APPSUPPORT_OVERLAY_LOOKUP="/opt/appsupport_overlay/
                           /home/appsupport_overlay/
                           /home/.appsupport_overlay/
                           /home/
                           /home/.
                           /home/defaultuser/."
APPSUPPORT_OVERLAY_LOOKUP_FS="/opt/appsupport_overlay_fs
                              /home/appsupport_overlay_fs
                              /home/.appsupport_overlay_fs"

RUN_FROM_INIT=1
# When run from init SYSTEM_IMAGE_PATH and ROOTFS_PATH are set.
if [ -z "$SYSTEM_IMAGE_PATH" ] || [ -z "$ROOTFS_PATH" ]; then
    SYSTEM_IMAGE_PATH="$(appsupport-config --system)"
    RUN_FROM_INIT=0
fi

OVERLAY_SOURCE=
OVERLAY_SOURCE_TYPE=
OVERLAY_IMG=
OVERLAY_ROOT=
OVERLAY_SOURCE_ACCESS=
SYSTEM_IMG_ACCESS=

APPSUPPORT_WAS_RUNNING=0
FORCE_INSTALL=0
UNINSTALL=0

SYSTEM_IMG_MOUNTED=0
SYSTEM_IMG_MOUNT=/tmp/tmp_appsupport_mount.$$

if [ "$(whoami)" != "root" ]; then
    echo "This script needs to be run by root."
    exit 1
fi

if [ -z "$SYSTEM_IMAGE_PATH" ] || [ ! -f "$SYSTEM_IMAGE_PATH" ]; then
    echo "Could not determine system image path!!!"
    exit 1
fi

UNPRIVILEGED_CONTAINER=$(appsupport-config -g ro.alien.unprivileged)

overlay_mount_op() {
    _OP="$1"

    if [ $RUN_FROM_INIT -eq 0 ]; then
        echo "$_OP operation cannot be run outside init"
        exit 1
    fi

    if [ ! -f "$APPSUPPORT_OVERLAY_CACHE" ]; then
        echo "No overlay cache in $APPSUPPORT_OVERLAY_CACHE !"
        exit 1
    fi

    source "$APPSUPPORT_OVERLAY_CACHE"

    if [ "$USE_OVERLAY_FS" = "n" ]; then
        exit 100
    fi

    OVERLAY_ROOT_FS="$OVERLAY_ROOT/overlay"

    if [ "$OVERLAY_SOURCE_TYPE" = "fs" ]; then
        OVERLAY_ROOT_FS="$OVERLAY_SOURCE"
    fi

    case "$_OP" in
        mount)
            if [ "$USE_OVERLAY_FS" = "m" ]; then
                modprobe overlay
            fi
            if [ "$OVERLAY_SOURCE_TYPE" = "img" ]; then
                mkdir -p "$OVERLAY_ROOT_FS"
                mount -o loop "$OVERLAY_SOURCE" "$OVERLAY_ROOT_FS"
            fi
            # First mount system.img
            mkdir -p "$OVERLAY_ROOT/rootfs"
            mount -o loop "$SYSTEM_IMAGE_PATH" "$OVERLAY_ROOT/rootfs"
            mkdir -p "$ROOTFS_PATH"
            mount overlay -t overlay -o lowerdir="$OVERLAY_ROOT/rootfs":"$OVERLAY_ROOT_FS" "$ROOTFS_PATH"
            exit $?
            ;;
        umount)
            umount "$ROOTFS_PATH"
            umount "$OVERLAY_ROOT/rootfs"
            if [ "$OVERLAY_SOURCE_TYPE" = "img" ]; then
                umount "$OVERLAY_ROOT_FS"
            fi
            exit 0
            ;;
        *)
            echo "Invalid overlay op $_OP"
            exit 1
            ;;
    esac
}

if [ "$1" = "force" ]; then
    FORCE_INSTALL=1
elif [ "$1" = "uninstall" ]; then
    UNINSTALL=1
elif [ "$1" = "mount" ]; then
    overlay_mount_op mount
elif [ "$1" = "umount" ]; then
    overlay_mount_op umount
elif [ -n "$1" ]; then
    echo "Invalid overlay op $1"
    exit 1
fi

if [ $RUN_FROM_INIT -eq 0 ] && systemctl is-active appsupport > /dev/null; then
    APPSUPPORT_WAS_RUNNING=1
fi

# Check for already installed overlay
if [ -f "$APPSUPPORT_OVERLAY_CACHE" ]; then
    source "$APPSUPPORT_OVERLAY_CACHE"

    if [ -z "$SYSTEM_IMG_ACCESS" ] ||
       [ -z "$OVERLAY_SOURCE_ACCESS" ] ||
       [ "$APPSUPPORT_OVERLAY_VERSION" != "$OVERLAY_VERSION" ] ||
       [ ! -e "$OVERLAY_SOURCE" ] ||
       [ "$SYSTEM_IMG_ACCESS" != "$(stat -c '%Y' $SYSTEM_IMAGE_PATH)" ] ||
       [ "$OVERLAY_SOURCE_ACCESS" != "$(stat -c '%Y' "$OVERLAY_SOURCE")" ]; then
        FORCE_INSTALL=1
    fi

    if [ $FORCE_INSTALL -eq 1 ]; then
        if [ $APPSUPPORT_WAS_RUNNING -eq 1 ]; then
            systemctl stop appsupport
        fi
        if [ -d "$OVERLAY_ROOT" ]; then
            rm -rf "$OVERLAY_ROOT"
        fi
        if [ ! -e "$OVERLAY_SOURCE" ]; then
            OVERLAY_SOURCE=
        fi
        OVERLAY_SOURCE_ACCESS=
        SYSTEM_IMG_ACCESS=
        rm -f "$APPSUPPORT_OVERLAY_CACHE"
        rm -f "$APPSUPPORT_OVERLAY_CONFIG"
    else
        # Overlay ok
        if [ $UNINSTALL -eq 1 ]; then
            if [ $APPSUPPORT_WAS_RUNNING -eq 1 ]; then
                systemctl stop appsupport
            fi

            if [ -d "$OVERLAY_ROOT" ]; then
                rm -rf "$OVERLAY_ROOT"
            fi
            rm -f "$APPSUPPORT_OVERLAY_CACHE"
            rm -f "$APPSUPPORT_OVERLAY_CONFIG"

            if [ $APPSUPPORT_WAS_RUNNING -eq 1 ]; then
                systemctl restart appsupport
            fi
        fi
        exit 0
    fi
fi

if [ -z "$OVERLAY_SOURCE" ]; then
    LOOKUP_IMG_FOUND=0
    # Generic locations
    for LOOKUP in $APPSUPPORT_OVERLAY_LOOKUP; do
        for LOOKUP_IMG in $APPSUPPORT_OVERLAY_IMG_UNPRIVILEGED $APPSUPPORT_OVERLAY_IMG_PRIVILEGED; do
            if [ -f "$LOOKUP$LOOKUP_IMG" ]; then
                LOOKUP_IMG_FOUND=1
                OVERLAY_SOURCE="$LOOKUP$LOOKUP_IMG"
                OVERLAY_SOURCE_TYPE="img"
                OVERLAY_ROOT="$(dirname $OVERLAY_SOURCE)/.appsupport_overlay"
                break
            fi
        done
        if [ $LOOKUP_IMG_FOUND -eq 1 ]; then
            break;
        fi

        if [ -f "$LOOKUP$APPSUPPORT_OVERLAY_TAR" ]; then
            OVERLAY_SOURCE="$LOOKUP$APPSUPPORT_OVERLAY_TAR"
            OVERLAY_SOURCE_TYPE="tar"
            OVERLAY_ROOT="$(dirname $OVERLAY_SOURCE)/.appsupport_overlay"
            break
        fi
    done

    # Flat file system locations
    if [ -z "$OVERLAY_SOURCE" ]; then
        for LOOKUP in $APPSUPPORT_OVERLAY_LOOKUP_FS; do
            if [ -d "$LOOKUP" ]; then
                OVERLAY_SOURCE="$LOOKUP"
                OVERLAY_SOURCE_TYPE="fs"
                OVERLAY_ROOT="$(dirname $OVERLAY_SOURCE)/.appsupport_overlay"
                break
            fi
        done
    fi
fi

if [ -z "$OVERLAY_SOURCE" ]; then
    # No overlay source.
    exit 0
fi

overlay_done_exit() {
    if [ $SYSTEM_IMG_MOUNTED -eq 1 ]; then
        umount "$SYSTEM_IMG_MOUNT"
    fi
    if [ -d "$SYSTEM_IMG_MOUNT" ]; then
        rmdir "$SYSTEM_IMG_MOUNT"
    fi
}

trap overlay_done_exit EXIT

#
# Check for OVERLAY_FS support
#

USE_OVERLAY_FS="n"

if [ -f "/proc/config.gz" ]; then
    CONFIG_CHECK="gunzip -c /proc/config.gz"
elif [ -f "/proc/config" ]; then
    CONFIG_CHECK="cat /proc/config"
fi

if [ -n "$CONFIG_CHECK" ]; then
    $CONFIG_CHECK | grep -q "^CONFIG_OVERLAY_FS=y"
    if [ $? -eq 0 ]; then
        USE_OVERLAY_FS="y"
    else
        $CONFIG_CHECK | grep -q "^CONFIG_OVERLAY_FS=m"
        if [ $? -eq 0 ]; then
            modprobe overlay
            if [ $? -eq 0 ]; then
                USE_OVERLAY_FS="m"
            fi
        fi
    fi
fi

if [ "$OVERLAY_SOURCE_TYPE" = "fs" ] && [ "$USE_OVERLAY_FS" = "n" ]; then
    echo "Flat filesystem source cannot be used without OVERLAY_FS."
    exit 1
fi

mkdir $SYSTEM_IMG_MOUNT
if [ $? -ne 0 ]; then
    echo "Failed to create location for system image mount $SYSTEM_IMG_MOUNT"
    exit 1
fi

if [ "$USE_OVERLAY_FS" = "n" ]; then
    mount $SYSTEM_IMAGE_PATH $SYSTEM_IMG_MOUNT
    if [ $? -eq 0 ]; then
        SYSTEM_IMG_MOUNTED=1
    else
        echo "Failed to mount $SYSTEM_IMAGE_PATH to $SYSTEM_IMG_MOUNT"
        exit 1
    fi
fi

OVERLAY_ROOT_FS="$OVERLAY_ROOT/overlay"
mkdir -p $OVERLAY_ROOT_FS
mkdir -p $(dirname $APPSUPPORT_OVERLAY_CACHE)
echo "# appsupport overlay, do not edit, this file is autogenerated." > $APPSUPPORT_OVERLAY_CONFIG

if [ "$OVERLAY_SOURCE_TYPE" = "fs" ]; then
    OVERLAY_ROOT_FS="$OVERLAY_SOURCE"
fi

#
# When no overlayfs support is available copy full directory contents of
# overlay components.
#

if [ "$USE_OVERLAY_FS" = "n" ]; then
    if [ "$OVERLAY_SOURCE_TYPE" = "img" ]; then
        TMP_DIR="/tmp/appsupport-overlay-img-tmp.$$"
        mkdir -p "$TMP_DIR"
        mount -o loop "$OVERLAY_SOURCE" "$TMP_DIR"
        cd "$TMP_DIR"
        for OVERLAY_PART in $(find . -mindepth 2 -maxdepth 2 -type d | cut -d'/' -f2-); do
            if [ "$OVERLAY_PART" = "init.sh" ]; then
                continue
            fi
            mkdir -p $(dirname "$OVERLAY_ROOT_FS/$OVERLAY_PART")
            cp -a $SYSTEM_IMG_MOUNT/$OVERLAY_PART $OVERLAY_ROOT_FS/$OVERLAY_PART

            echo "lxc.mount.entry = $OVERLAY_ROOT_FS/$OVERLAY_PART $OVERLAY_PART none rbind,ro 0 0" >> $APPSUPPORT_OVERLAY_CONFIG
        done
        cp -a * "$OVERLAY_ROOT_FS"
        cd /
        umount "$TMP_DIR"
        rmdir "$TMP_DIR"
    fi

    if [ "$OVERLAY_SOURCE_TYPE" = "tar" ]; then
        for OVERLAY_PART in $(gunzip -c "$OVERLAY_SOURCE" | tar --list | cut -d'/' -f-2 | sort | uniq); do
            if [ "$(echo $OVERLAY_PART | cut -d'/' -f2)" = "" ]; then
                mkdir -p "$OVERLAY_ROOT_FS/$OVERLAY_PART"
                ORIG_OWNERSHIP=$(stat -c "%u:%g" "$SYSTEM_IMG_MOUNT/$OVERLAY_PART")
                ORIG_PERMISSIONS=$(stat -c "%a" "$SYSTEM_IMG_MOUNT/$OVERLAY_PART")
                chown $ORIG_OWNERSHIP "$OVERLAY_ROOT_FS/$OVERLAY_PART"
                chmod $ORIG_PERMISSIONS "$OVERLAY_ROOT_FS/$OVERLAY_PART"
                continue
            fi

            if [ "$OVERLAY_PART" = "init.sh" ]; then
                continue
            fi

            cp -a $SYSTEM_IMG_MOUNT/$OVERLAY_PART $OVERLAY_ROOT_FS/$OVERLAY_PART

            echo "lxc.mount.entry = $OVERLAY_ROOT_FS/$OVERLAY_PART $OVERLAY_PART none rbind,ro 0 0" >> $APPSUPPORT_OVERLAY_CONFIG
        done
    fi
fi

if [ "$OVERLAY_SOURCE_TYPE" = "tar" ]; then
    tar oxzf $OVERLAY_SOURCE -C $OVERLAY_ROOT_FS
fi

if [ "$UNPRIVILEGED_CONTAINER" = "1" ]; then
    ANDROID_BASE_UID=$(appsupport-config -g ro.alien.unprivileged_uid)
    TMP_FILES="/tmp/appsupport_overlay.$$.$RANDOM"

    find "$OVERLAY_ROOT_FS" \! -type l -mindepth 1 > $TMP_FILES
    if [ $? -ne 0 ]; then
        echo "Appsupport overlay: Shifting $OVERLAY_ROOT_FS files up failed (find failed)." >&2
        rm -rf "$OVERLAY_ROOT_FS"
        exit 1
    fi

    $(appsupport-config --shift) up $ANDROID_BASE_UID < $TMP_FILES
    if [ $? -ne 0 ]; then
        echo "Appsupport overlay: Shifting $OVERLAY_ROOT_FS files up failed (shift_ids failed)." >&2
        rm -rf "$OVERLAY_ROOT_FS"
        exit 1
    fi

    rm -f $TMP_FILES
fi

# Move init script to correct location
if [ -f "$OVERLAY_ROOT_FS/init.sh" ]; then
    mv "$OVERLAY_ROOT_FS/init.sh" "$OVERLAY_ROOT"
fi

RUN_OVERLAY_INIT=

if [ -f "$OVERLAY_ROOT/init.sh" ]; then
    RUN_OVERLAY_INIT="$OVERLAY_ROOT/init.sh"
fi

if [ "$OVERLAY_SOURCE_TYPE" = "img" ]; then
    TMP_DIR="/tmp/appsupport-overlay-img-tmp.$$"
    mkdir -p "$TMP_DIR"
    mount -o loop "$OVERLAY_SOURCE" "$TMP_DIR"
    if [ -f "$TMP_DIR/.overlay-init.sh" ]; then
        RUN_OVERLAY_INIT="${TMP_DIR}-overlay-init.sh"
        cp "$TMP_DIR/.overlay-init.sh" "$RUN_OVERLAY_INIT"
    fi
    umount "$TMP_DIR"
fi

if [ "$OVERLAY_SOURCE_TYPE" = "fs" ]; then
    if [ -f "$OVERLAY_SOURCE/.overlay-init.sh" ]; then
        RUN_OVERLAY_INIT="${OVERLAY_SOURCE}/.overlay-init.sh"
    fi
fi

if [ -n "$RUN_OVERLAY_INIT" ]; then
    OVERLAY_CONFIG=$APPSUPPORT_OVERLAY_CONFIG OVERLAY_ROOT=$OVERLAY_ROOT SYSTEM_IMG_MOUNT=$SYSTEM_IMG_MOUNT bash "$RUN_OVERLAY_INIT"
fi

cat << EOF >$APPSUPPORT_OVERLAY_CACHE
OVERLAY_VERSION=$APPSUPPORT_OVERLAY_VERSION
SYSTEM_IMG_ACCESS=$(stat -c "%Y" $SYSTEM_IMAGE_PATH)
OVERLAY_SOURCE_ACCESS=$(stat -c "%Y" $OVERLAY_SOURCE)
OVERLAY_SOURCE_TYPE=$OVERLAY_SOURCE_TYPE
OVERLAY_SOURCE=$OVERLAY_SOURCE
OVERLAY_ROOT=$OVERLAY_ROOT
USE_OVERLAY_FS=$USE_OVERLAY_FS
EOF

if [ $APPSUPPORT_WAS_RUNNING -eq 1 ]; then
    systemctl restart appsupport
fi
